
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <wiringPi.h>
#include <wiringPiI2C.h>
#include <math.h>
#include <time.h>

#define VREF 5.0f           
#define SCOUNT 30           // Sampling points
#define TEMPERATURE 25.0f   // Water temperature (Celsius)

int analogBuffer[SCOUNT];        
int analogBufferTemp[SCOUNT];
int analogBufferIndex = 0;
float averageVoltage = 0, tdsValue = 0;

int ads1115_fd;
// Function declarations
int getMedianNum(int bArray[], int iFilterLen);
int readADS1115();
unsigned long my_millis();  

unsigned long my_millis() {
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return (ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
}

int getMedianNum(int bArray[], int iFilterLen) {
    int bTab[iFilterLen];
    for (int i = 0; i < iFilterLen; i++) {
        bTab[i] = bArray[i];
    }
    int i, j, bTemp;
    for (j = 0; j < iFilterLen - 1; j++) {
        for (i = 0; i < iFilterLen - j - 1; i++) {
            if (bTab[i] > bTab[i + 1]) {
                bTemp = bTab[i];
                bTab[i] = bTab[i + 1];
                bTab[i + 1] = bTemp;
            }
        }
    }
    
    if ((iFilterLen & 1) > 0) {
        bTemp = bTab[(iFilterLen - 1) / 2];
    } else {
        bTemp = (bTab[iFilterLen / 2] + bTab[iFilterLen / 2 - 1]) / 2;
    }
    return bTemp;
}

int readADS1115() {
    // Configure ADS1115: Channel 0
    uint16_t config = 0xC283;  
    wiringPiI2CWriteReg16(ads1115_fd, 0x01, (config >> 8) | (config << 8));
    usleep(8000);  
    
    uint16_t result = wiringPiI2CReadReg16(ads1115_fd, 0x00);
    int16_t adc = ((result >> 8) & 0xFF) | ((result & 0xFF) << 8);
    
 
    static int debug_count = 0;
    if (debug_count++ % 20 == 0) {
        printf("[Raw ADC: %d] ", adc);
    }
    
    int adc_abs = abs(adc);
    // Convert to analog value (0-1023)
    int arduino_value = (adc_abs * 1024) / 30000;
    // Limit to 0-1023 range
    if (arduino_value > 1023) arduino_value = 1023;
    if (arduino_value < 0) arduino_value = 0;
    return arduino_value;
}
int main() {
    printf("=== Raspberry Pi TDS Measurement ===\n");
    printf("TDS = 133.42*V^3 - 255.86*V^2 + 857.39*V * 0.5\n\n");
    
    // Initialize wiringPi
    if (wiringPiSetup() < 0) {
        printf("wiringPi initialization failed\n");
        return 1;
    }
    
    // Initialize ADS1115
    ads1115_fd = wiringPiI2CSetup(0x48);
    if (ads1115_fd < 0) {
        printf("ADS1115 initialization failed (address 0x48)\n");
        printf("Please check: sudo i2cdetect -y 1\n");
        return 1;
    }
    printf("✓ Hardware initialization completed\n");
    printf("Starting measurement...\n");
    printf("----------------------------------------\n");
    
    // Timepoint variables (using custom my_millis)
    unsigned long analogSampleTimepoint = my_millis();
    unsigned long printTimepoint = my_millis();
    
    while (1) {
        unsigned long currentMillis = my_millis();     
        if (currentMillis - analogSampleTimepoint > 40U) {
            analogSampleTimepoint = currentMillis;
            
            int arduinoValue = readADS1115();
            
            analogBuffer[analogBufferIndex] = arduinoValue;
            analogBufferIndex++;
            if (analogBufferIndex == SCOUNT) {
                analogBufferIndex = 0;
            }
        }
         
        if (currentMillis - printTimepoint > 800U) {
            printTimepoint = currentMillis;
           
            for (int copyIndex = 0; copyIndex < SCOUNT; copyIndex++) {
                analogBufferTemp[copyIndex] = analogBuffer[copyIndex];
            }
   
            int medianValue = getMedianNum(analogBufferTemp, SCOUNT);
            averageVoltage = medianValue * VREF / 1024.0;
            float compensationCoefficient = 1.0 + 0.02 * (TEMPERATURE - 25.0);
            float compensationVoltage = averageVoltage / compensationCoefficient;
            tdsValue = (133.42 * compensationVoltage * compensationVoltage * compensationVoltage 
                       - 255.86 * compensationVoltage * compensationVoltage 
                       + 857.39 * compensationVoltage) * 0.5;
            printf("TDS Value: %.0fppm", tdsValue);
           
            printf(" (ADC:%d, Volt:%.3fV)\n", medianValue, averageVoltage);
            
            fflush(stdout);
        }
        
        usleep(1000);  
    }
    
    return 0;
}
